﻿//---------------------------------------------------------------------------------
// Microsoft (R) .NET Services SDK
// Software Development Kit
// 
// Copyright (c) Microsoft Corporation. All rights reserved.  
//
// THIS CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND, 
// EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES 
// OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE. 
//---------------------------------------------------------------------------------

namespace ProAzure.MachineInfo.Service
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Security.Cryptography;
    using System.Text;
    using System.Web;

    public class TokenFactory
    {
        string acsHost;
        string solutionName;
        string issuerName;
        string signingKey;

        public TokenFactory(string acsHost, string solutionName, string issuerName, string signingKey)
        {
            this.acsHost = acsHost;
            this.solutionName = solutionName;
            this.issuerName = issuerName;
            this.signingKey = signingKey;
        }

        public string CreateToken(Dictionary<string, string> claims)
        {
            // check for dup claimtypes
            Dictionary<string, string> claimList = this.RemoveDuplicateClaimTypes(claims);
            
            // build the claims string
            StringBuilder builder = new StringBuilder();
            foreach (KeyValuePair<string, string> entry in claimList)
            {
                builder.Append(entry.Key);
                builder.Append('=');
                builder.Append(entry.Value);
                builder.Append('&');
            }

            // add the issuer name
            builder.Append("Issuer=");
            builder.Append(this.issuerName);
            builder.Append('&');

            // add the Audience
            builder.Append("Audience=");
            builder.Append(string.Format("https://{0}.{1}/WRAPv0.8&", this.solutionName, this.acsHost));

            // add the expires on date
            builder.Append("ExpiresOn=");
            builder.Append(GetExpiresOn(20));

            string signature = this.GenerateSignature(builder.ToString(), this.signingKey);
            builder.Append("&HMACSHA256=");
            builder.Append(signature);

            return builder.ToString();
        }

        private string GenerateSignature(string unsignedToken, string signingKey)
        {
            HMACSHA256 hmac = new HMACSHA256(Convert.FromBase64String(signingKey));

            byte[] locallyGeneratedSignatureInBytes = hmac.ComputeHash(Encoding.ASCII.GetBytes(unsignedToken));

            string locallyGeneratedSignature = HttpUtility.UrlEncode(Convert.ToBase64String(locallyGeneratedSignatureInBytes));

            return locallyGeneratedSignature;
        }

        private static ulong GetExpiresOn(double minutesFromNow)
        {
            TimeSpan expiresOnTimeSpan = TimeSpan.FromMinutes(minutesFromNow);

            DateTime expiresDate = DateTime.UtcNow + expiresOnTimeSpan;

            TimeSpan ts = expiresDate - new DateTime(1970, 1, 1, 0, 0, 0, 0);

            return Convert.ToUInt64(ts.TotalSeconds);
        }

        private Dictionary<string, string> RemoveDuplicateClaimTypes(Dictionary<string, string> claims)
        {
            Dictionary<string, string> newClaims = new Dictionary<string, string>();

            foreach (KeyValuePair<string, string> entry in claims)
            {
                string value;

                if (newClaims.TryGetValue(entry.Key, out value))
                {
                    newClaims[entry.Key] = value + "," + entry.Value;
                }
                else
                {
                    newClaims.Add(entry.Key, entry.Value);
                }
            }

            return newClaims;
        }
    }
}
